//
//  OPDatabase.h
//  OnePasswordiOS
//
//  Created by Roustem Karimov on 12-02-20.
//  Copyright (c) 2012 AgileBits. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <OnePasswordDataModel/OnePasswordDataModel.h>
#import "OPDatabase.h"

NS_ASSUME_NONNULL_BEGIN

@interface OP4Database : OPDatabase <OPDatabaseBasicItemSupport,
	OPDatabaseProfileSupport,
	OPDatabaseCategoriesSupport,
	OPDatabaseTransactionIdentificationSupport,
	OPDatabaseOptionalFolderSupport,
	OPDatabaseOptionalAttachmentSupport,
	OPDatabaseOptionalTraditionalSyncSupport,
	OPDatabaseOptionalEmptyTrashSupport>

@property (nonatomic, readonly) NSURL * databaseURL;
@property (nonatomic, readonly) NSURL * externalAttachmentDir;

@property (nonatomic, readonly) NSString * databaseUUID;
@property (nonatomic, readonly) NSInteger databaseAge;

@property (nonatomic, strong, nullable) OPProfile * activeProfile;
@property (nonatomic, strong, nullable) OPProfile * primaryProfile;

@property (nonatomic, strong, nullable) NSString * txSyncKeychainIdentifier; // Both identifies the currently running syncer and the fact that this transaction is performed during sync

+ (OP4Database *)activeDatabase;
+ (void)setActiveDatabase:(nullable OP4Database *)database;
+ (void)destroyActiveDatabase;

- (id)initWithURL:(NSURL *)URL;

/**
* Synchronous method used in unit tests.
*/
- (BOOL)openDatabaseWithError:(NSError **)error;

/**
 * The offline operation will be performed when the SQLite file is closed. The SQLite file will be reopen and caches cleared after the operation is completed.
 * If before: block is not nil and returns NO then the offline operation is not performed.
 */
- (OPDatabaseOperation *)performOfflineOperation:(BOOL (^)(OP4Database *db, NSError **error))offlineOperation before:(nullable BOOL (^)(OP4Database *db, BOOL *rollback))before after:(nullable void (^)(BOOL success, NSError *error))after;

- (void)performSyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction;
- (void)performAsyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction completion:(nullable void (^)(BOOL success))completion;
- (void)performAsyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction;

- (void)performLowPriorityAsyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction completion:(void (^)(BOOL success))completion;
- (void)performLowPriorityAsyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction;
- (void)performLowPrioritySyncTransaction:(void (^)(OP4Database *db, BOOL *rollback))transaction;

- (NSArray *)allProfiles;
- (NSArray *)allUnlockedProfiles;
- (NSString *)uniqueProfileNameForName:(NSString *)profileName;

//
// The txSelect and txUpdate methods below must always be called within the performSyncTransaction or performAsyncTransaction
//
- (NSString *)txSelectConfigValueForKey:(NSString *)key;
- (BOOL)txSetConfigValue:(NSString *)value forKey:(NSString *)key;

- (BOOL)unlockProfile:(OPProfile *)profile withPassword:(NSString *)password error:(NSError **)error;
- (void)unlockSecondaryProfiles;
- (void)lockAllExceptThisProfile:(OPProfile *)profile;

- (OPProfile *)profileWithId:(OPID)profileId;
- (OPProfile *)profileWithUUID:(NSString *)profileUUID;
- (OPProfile *)profileWithName:(NSString *)profileName;

- (BOOL)txUpdateItem:(OPItem *)item mergeIfNecessary:(BOOL)mergeIfNecessary error:(NSError **)error;
- (BOOL)txUpdateProfile:(OPProfile *)profile error:(NSError **)error;
- (BOOL)txDeleteProfileWithId:(OPID)profileId error:(NSError **)error;

- (NSArray *)txSelectSmartFoldersForProfile:(OPProfile *)profile;


- (NSSet *)txProfile:(OPProfile *)profile selectItemUUIDsInBand:(NSString *)band;
- (BOOL)txUpdateTxTimestamp:(OPTimestamp)timestamp forItem:(OPItem *)item;


// Backup
- (NSUInteger)txSelectTotalProfileCount;
- (NSUInteger)txSelectTotalItemCountExcludingTombstonesInAllProfiles;
- (NSUInteger)txSelectTotalNonTrashedFolderCountInAllProfiles;
- (NSUInteger)txSelectTotalItemAttachmentCountInAllProfiles;

- (NSInteger)txProfileCountFavorites:(id<OPGenericProfile>)profile;
- (NSInteger)txProfileCountTrashed:(id<OPGenericProfile>)profile;
- (NSInteger)txProfileCountTombstones:(id<OPGenericProfile>)profile;
- (NSInteger)txProfileCountAttachments:(id<OPGenericProfile>)profile;
- (NSInteger)txProfileCountAllItems:(id<OPGenericProfile>)profile;

- (NSUInteger)txSelectMaxItemId;
- (NSArray *)txSelectAllItemUUIDsForProfile:(OPProfile *)profile;
- (NSArray *)txSelectAllItemUUIDsExcludingTombstonesForProfile:(OPProfile *)profile;
- (NSArray *)txSelectAllItemUUIDsExcludingTombstonesAndTrashedForProfile:(OPProfile *)profile;


- (NSArray *)txSelectOldTombstonesForProfile:(OPProfile *)profile;


- (NSArray *)txSelectAllSyncSources;
- (OPSyncSource *)txSelectSyncSourceForProfile:(id<OPGenericProfile>)profile;
- (BOOL)txProfile:(OPProfile *)profile deleteSyncSourceWithError:(NSError **)error;
- (BOOL)txDeleteAllSyncSourcesWithError:(NSError **)error;
- (BOOL)txUpdateSyncSource:(OPSyncSource *)syncSource error:(NSError **)error;

- (void)txRefreshSyncSourceForImport:(OPSyncSource *)syncSource;
- (void)txRefreshSyncSourceForExport:(OPSyncSource *)syncSource;
- (NSArray *)txSelectSyncStateForItem:(OPItem *)item;

- (OPSyncState *)txSelectSyncStateForItemId:(OPID)itemId syncSourceId:(OPID)syncSourceId;
- (OPSyncState *)txSelectSyncStateForFolderId:(OPID)folderId syncSourceId:(OPID)syncSourceId;

- (BOOL)txUpdateSyncState:(OPSyncState *)syncState error:(NSError **)error;
- (BOOL)txClearSyncSource:(OPSyncSource *)syncSource error:(NSError **)error;

//
// Utility methods
//
- (BOOL)txClearAutosaveTokensWithError:(NSError **)error;

// Backup
- (BOOL)backupToURL:(NSURL *)backupURL;

/// Must not be run within a transaction
- (BOOL)vacuum:(NSError **)error;

// Trial
- (NSInteger)daysSinceOnePassword5Migration;

/// You don't want to use these, trust me
- (id)_txSelectItemWithId:(OPID)oid loadCompletely:(BOOL)loadCompletely assertIfNotFound:(BOOL)assertIfNotFound;

@end

NS_ASSUME_NONNULL_END
